/**********************************************************************
 *                                                                    *
 * This C language library provides some simple matrix operations and *
 * the functions MLOAD and MSAVE enables the use of MAT files         *
 * generated by MATLAB.                                               *
 *                                                                    *
 * Coded by     : Steffen Torp & Magnus Norgaard                      *
 *                                                                    *
 * References   : Brugervejledning til C/Matlab matrix bibliotek,     *
 *                Appendix F i Noergaard og Torp (1993).              *
 *                                                                    *
 * Dependencies : matrix.h                                            *
 *                                                                    *
 * Comments     : All matrices are represented by a structure with    *
 *                the dimensions of the matrix and a pointer to       *
 *                an array of pointers to each row in the matrix.     *
 *                The functions using stdio can be excluded by        *
 *                defining the symbol NO_IO to the compiler.          *
 *                If the library is compiled on the OS/9 system       *
 *                the symbol OS9 must be defined.                     *
 *                                                                    *
 * Last edit    : sept. 22, 1994 Magnus Norgaard                      *
 *                                                                    *
 **********************************************************************/

#include <stdlib.h>
#include <string.h>

#include "matrix2.h"     

#define TRUE 1
#define FALSE 0

#ifndef NO_IO  /* I/O functions are not available in the DSP C compiler */
#include <stdio.h>
int  loadmat(FILE*, int*, char*, int*, int*, int*, double**, double**);
void savemat(FILE*,  int, char*,  int,  int,  int,  double*,  double*);
long longconv(long);           /* Internal functions, only used by  */
void matconv(int, double*);    /* The functions in this library.    */
typedef struct {               /* Matrix structure for MATLAB files */
	long type;
	long mrows;
	long ncols;
	long imagf;
	long namlen;
} Fmatrix;
#endif


#define READ "rb"
#define APND "ab"
#define SYSTEM 0               /* 0 = PC, 1000 = Apollo and OS/9 */
#ifdef OS9
  #undef READ
  #undef APND
  #undef SYSTEM
  #define SYSTEM 1000
  #define READ "r"
  #define APND "a"
#endif


/*
 * MMAKE :
 * -------
 * This function allocates memory for a matrix of the specified size
 * and assigns the specified dimensions to the allocated matrix.
 * Inputs  : rows - number of rows.
 *           cols - number of columns.
 * Outputs : *ptm - pointer to the new matrix structure.
 */
matrix *mmake( int rows, int cols )
{
	matrix *ptm;
	double **row_pointers;
	register int i;

#if RUNCHK

	/* Check that no dimension is zero. */
	if ((rows==0)||(cols==0)) merror("Invalid dimension error in mmake!");
#endif

	/* Memory allocation for matrix structure. */
	ptm = (matrix*)malloc(sizeof(matrix));

	/* Memory for array of pointers to rows. */;
	row_pointers = (double**)malloc(rows*sizeof(double*));

	/* Memory for all rows, initialize row pointers. */
	row_pointers[0] = (double*)malloc(rows*cols*sizeof(double));
	for (i=1;i<rows;i++) {
		row_pointers[i] = row_pointers[i-1] + cols;
	}

#if RUNCHK

	/* Check if last allocation was ok ! */
	if (!row_pointers[0]) {
		merror("Memory allocation error in mmmake!");
	}
#endif

	ptm->row = rows;             /* Initialize matrix structure */
	ptm->col = cols;
	ptm->mat = row_pointers;     /* Pointer to row pointers     */

	return ptm;           /* Return pointer to matrix structure */
}


/*
 * MFREE :
 * -------
 * This function deallocates the memory that was used for a matrix.
 * Input : Pointer to the matrix to be deallocated.
 */
void mfree( matrix *ptm )
{
	/* Deallocate rows */
	free(ptm->mat[0]);

	/* Deallocate row pointer array */
	free(ptm->mat);

	/* Deallocate matrix structure */
	free(ptm);
}



/*
 * MSAVE :
 * -------
 * This function is used to save a matrix in a MAT-file, that can be read
 * by MATLAB. The function uses the SAVEMAT.C function provided by MATLAB.
 * The function only handles numrical matrices.
 * Inputs : ptm       - pointer to the matrix that will be saved.
 *          file_name - string with name of file to save in.
 *          mat_name  - string with name of matrix in MATLAB.
 */
void msave( matrix *ptm, char file_name[], char mat_name[] )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	FILE *fp;
	double *preal,*pimag;
	double *data;
	int type,imagf,mn,i,j,k;

	mn = ptm->row*ptm->col;                       /* Number of data.  */
	data = (double*)malloc(mn*sizeof(double));    /* Allocate buffer. */
	k = 0;

	/* Prepare for SAVEMAT call using MATLAB format. */

	for ( j=0; j < ptm->col; j++ )
	{
		for ( i=0; i < ptm->row; i++ )
		{
			*(data+k) = get_val(ptm,i,j); /* Write to buffer. */
			k++;
		}
	}
	imagf = 0;                  /* No complex data.                   */
	type  = SYSTEM;             /* Only numeric data.                 */
	preal = data;               /* Real data is in buffer "data".     */
	pimag = (double *)0;        /* No complex data.                   */
	fp = fopen(file_name,APND); /* Append to existing file or create. */
				    /* new one, using binary file-format. */ 
	if (!(fp)) merror("Can't open file error in msave!");
	savemat(fp,type,mat_name,ptm->row,ptm->col,imagf,preal,pimag);
				    /* Save the matrix in MAT-file.       */
	fclose(fp);                 /* Close MAT-file.                    */
	free(data);                 /* Deallocate buffer memory.          */
#else
	ptm=ptm;
	file_name=file_name;
	mat_name=mat_name;
	exit(0); /* printf is not available in DSP compiler. */
#endif
}


/*
 * MLOAD :
 * -------
 * This function is used to load a matrix from a MAT-file, that was written
 * by MATLAB. The function uses the LOADMAT.C function provided by MATLAB,
 * but it only handles numerical and real matrices.
 * Inputs : file_name - string with the name of the MAT-file to read.
 *          mat_name  - string with the name of the matrix to load.
 * Output : ptm       - Pointer to the loaded matrix.
 */
matrix *mload( char file_name[], char mat_name[] )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	FILE *fp;
	matrix *ptm;
	char name[20];
	int found,i,j,k;
	int type,mrows,ncols,imagf;
	double *xr,*xi;

	fp = fopen(file_name,READ);        /* Open MAT-file for reading   */
					   /* from the beginning of file. */
	if (!(fp)) merror("File not found error in mload!");

	do  /* Search for the matrix with name in mat_name */
	{
		/* Read the file until matrix is found or EOF is reached. */
		if (loadmat(fp,&type,name,&mrows,&ncols,&imagf,&xr,&xi)) {
			printf("The searched matrix is : %s \n",mat_name);
			merror("matrix not in file error in mload!");
		}

		found = strncmp(mat_name,name,20);
	}
	while ( found!=0 );      /* Keep searching until matrix is found. */

	if (imagf) {
		printf("%s\n","Warning from mload :");
		printf("%s\n","The matrix was complex, but the");
		printf("%s\n","imaginary part has been ignored !");
	}

	if ((mrows==0)||(ncols==0)) merror("Matrix was empty error in mload!");

	ptm = mmake( mrows, ncols ); /* Allocate memory for loaded matrix */

	/* Copy data from buffer to matrix using C matrix structure. */
	k = 0;
	for ( j=0; j<ncols; j++ )
	{
		for ( i=0; i<mrows; i++ )
		{                               
			put_val(ptm,i,j,*(xr+k));
			k++;
		}
	}

	free(xr);   /* Deallocate data buffers and pointers.*/
	if (imagf) free(xi);
	fclose(fp); /* Close MAT-file.                      */
	return ptm; /* Return pointer to the loaded matrix. */
#else
	file_name=file_name;
	mat_name=mat_name;
	exit(0); /* printf is not available in DSP compiler. */
#endif
}


/*
 * SLOAD :
 * -------
 * This function loads a scalar (one by one matrix) from a MAT-file.
 * Inputs : file_name - Name of MAT-file to search for scalar.
 *          scal_name - Name of Matlab variable in MAT-file.
 * Output : scalar    - The scalar, which was loaded.
 */
double sload( char file_name[], char scal_name[] )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	matrix *ptm;
	double scalar;

	ptm    = mload( file_name, scal_name );
	scalar = ptm->mat[0][0];
	mfree(ptm);

	return scalar;
#else
	file_name=file_name;
	scal_name=scal_name;
	exit(0); /* printf is not available in DSP compiler. */
#endif
}


/*
 * MERROR :
 * --------
 * This function displays the specified error message and exits to server
 * operating system.
 * Input : Text string with the error message.
 */
void merror( char error_text[] )
{

#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	printf("--- Runtime error in matrix library:\n");
	printf("--- ");
	printf("%s\n\n",error_text);
#else
	error_text=error_text; /* Dummy command to avoid warnings on DSP */ 
#endif
	/*exit(1);*/
	mexErrMsgTxt("AN ERROR OCCURED IN A CMEX-PROGRAM");
}


/*
 * MPRINT :
 * --------
 * This is a simple function to print a matrix on the screen.
 * Input : ptm - Pointer to the matrix that will be printed.
 */
void mprint( matrix *ptm )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	int i,j;
	for ( i=0; i < ptm->row; i++ )
	{
		for( j=0; j < ptm->col; j++ )
		{
			printf("%4f %s",ptm->mat[i][j]," ");
		}
		printf("\n");
	}
#else
	ptm=ptm;
	exit(0); /* printf is not available in DSP compiler. */
#endif
}


/*
 * MINPUT1 :
 * ---------
 * This function reads the dimensions and content of a matrix from 
 * the keyboard.
 * Output : Pointer to new matrix.
 */
matrix *minput1()
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	matrix *ptm;
	int i,j,mrows,ncols;

	printf("%s","Number of rows    :");
	scanf("%d",&mrows);
	printf("%s","Number of columns :");
	scanf("%d",&ncols);

	ptm = mmake(mrows,ncols);

	for ( i=0; i<mrows; i++ )
	{
		for ( j=0; j<ncols; j++ )
		{
			printf("%s %d %s %d %s","Enter element (",i,",",j,")  :");
			scanf("%lf",&(ptm->mat[i][j]));
			/* PS. scanf does not work with floating formats in Turbo C. */
		}
	}
	return ptm;
#else
	exit(0); /* scanf is not available in DSP compiler. */
#endif
}


/*
 * MINPUT2 :
 * ---------
 * This function prompts for the content of a matrix of known size.
 * Input : Pointer to matrix to be read from keyboard.
 */
void minput2( matrix *ptm )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	int i,j;

	printf("%s %d %s %d %s\n","This matrix has",ptm->row,"rows and",
			ptm->col,"columns.");
	printf("%s %d %s\n","Please enter the following",
			ptm->row*ptm->col,"elements.");

	for ( i=0; i < ptm->row; i++ )
	{
		for ( j=0; j < ptm->col; j++ )
		{
			printf("%s %d %s %d %s","Enter element (",i,",",j,")  :");
			scanf("%lf",&(*ptm).mat[i][j]);
			/* PS. scanf does not work with floating formats in Turbo C. */
		}
	}
#else
	ptm=ptm;
	exit(0); /* scanf is not available in DSP compiler. */
#endif
}


/*
 * MTRANS :
 * --------
 * This function computes the transposed of a matrix.
 * Inputs : ptm2 - Pointer to argument matrix.
 *          ptm1 - Pointer to result matrix.
 * Notice : The two matrices may NOT be the same.
 */
void mtrans( matrix *ptm1, matrix *ptm2 )
{
	register int i,j;

#if RUNCHK

	if (ptm1==ptm2) merror("Memory conflict error in mtrans!");
#endif

	for ( i=0; i < ptm2->col; i++ )          /* Transpose matrix */
	{
		for ( j=0; j < ptm2->row; j++ )
		{
			ptm1->mat[i][j] = ptm2->mat[j][i];
		}
	}
}


/*
 * MINIT :
 * -------
 * This function initializes a matrix to all zeros.
 * Inputs : ptm - Pointer to the matrix that will be initialized.
 */
void minit( matrix *ptm )
{
	register unsigned char zero_byte = 0;
	register int bytes;
	
	/* Number of bytes in matrix data area. */
	bytes = 8*(ptm->row)*(ptm->col);

	/* Set data area to all zeroes. */
	memset( ptm->mat[0], zero_byte, bytes );
}



/*
 * MINITX :
 * --------
 * This function initializes a matrix to all x's. x being a scalar.
 * Inputs : ptm - Pointer to the matrix that will be initialized.
 *          x   - Element to be put into the matrix  
 */
void minitx( matrix *ptm, double x )
{
	register int i,j;
        for(i=0; i < ptm->row; i++)
	{
	  for(j=0; j < ptm->col; j++)
          {	
	    ptm->mat[i][j]=x;
	  }
	}
}


/*
 * MRAND :
 * -------
 * This function initializes a matrix with random numbers between -1 and 1.
 * Inputs : ptm - Pointer to the matrix that will be initialized.
 */
void mrand( matrix *ptm )
{
	register int i,j;
        for(i=0; i < ptm->row; i++){
	  for(j=0; j < ptm->col; j++)
	              ptm->mat[i][j]=rand()*2.0/RAND_MAX - 1.0;
	}
}



/*
 * MFIND :
 * -------
 * This function finds a given element in a matrix and returns
 * the row and coloumn coordinates to the locations. If no
 * instances of the element is found, a zero by zero matrix is
 * returned.
 * 
 * Inputs : ptm    - Pointer to matrix.
 *          elem   - Element to be searched for (double)
 * Output : retmat - 2 coloumn matrix. 1st coloumn contains row indices,
 *                   2nd coloumn contains the colomn indices.
 */
matrix *mfind( matrix *ptm, double elem )
{
        register int i,j,k;
	matrix *tmp, *retmat;
	k=0;
	tmp = mmake((ptm->row)*(ptm->col),2);

	/* Seach entire matrix */
	for(i=0; i < ptm->row; i++)
	  {
	    for(j=0; j < ptm->col; j++)
	      {
		if(ptm->mat[i][j] == elem)
		  {
		    tmp->mat[k][0]=i;
		    tmp->mat[k][1]=j;
                    k++;
		  }
	      }
	  }
	if(k==0)   /* The element is'nt recognized */
	{
	  retmat = mmake(1,1);
	  retmat->row=0;
	  retmat->col=0;
	}          /* Copy coordinates to a [ 2 | # of instances] matrix */
	else
	  {
	  retmat = mmake(k,2);
	  for(i=0; i < k; i++)
	  {
	    retmat->mat[i][0] = tmp->mat[i][0];
	    retmat->mat[i][1] = tmp->mat[i][1];
	  }
	}
	mfree(tmp);
	return retmat;
}



/*
 * MNOFIND :
 * ---------
 * This function finds all entries in a matrix not equal to a given element
 * and returns the row and coloumn coordinates to the locations. If all entries
 * equals the element, a zero by zero matrix is returned.
 * 
 * Inputs : ptm    - Pointer to matrix.
 *          elem   - Element to be searched for (double)
 * Output : retmat - 2 coloumn matrix. 1st coloumn contains row indices,
 *                   2nd coloumn contains the colomn indices.
 */
matrix *mnofind( matrix *ptm, double elem )
{
        register int i,j,k;
	matrix *tmp, *retmat;
	k=0;
	tmp = mmake((ptm->row)*(ptm->col),2);

	/* Seach entire matrix */
	for(i=0; i < ptm->row; i++)
	  {
	    for(j=0; j < ptm->col; j++)
	      {
		if(ptm->mat[i][j] != elem)
		  {
		    tmp->mat[k][0]=i;
		    tmp->mat[k][1]=j;
                    k++;
		  }
	      }
	  }
	if(k==0)   /* No entries different from the element */
	{
	  retmat = mmake(1,1);
	  retmat->row=0;
	  retmat->col=0;
	}          /* Copy coordinates to a [ 2 | # of instances] matrix */
	else
	  {
	  retmat = mmake(k,2);
	  for(i=0; i < k; i++)
	  {
	    retmat->mat[i][0] = tmp->mat[i][0];
	    retmat->mat[i][1] = tmp->mat[i][1];
	  }
	}
	mfree(tmp);
	return retmat;
}



/* 
 * MDIAG :
 * -------
 * This function initializes a matrix to all zeros except for the
 * main diagonal elements, which will be assigned the specified value.
 * Inputs : ptm - Pointer to matrix that will be initialized.
 *          diag - Diagonal element value.
 */
void mdiag( matrix *ptm, double diag )
{
	register int i;

#if RUNCHK

	if (ptm->row != ptm->col) \
	merror("matrix not quadratic error in mdiag!");

#endif

	minit(ptm);
	for ( i=0; i < ptm->row; i++ ) ptm->mat[i][i] = diag;
}


/*
 * MUNIT :
 * -------
 * This function initializes a matrix to the unit or identity matrix.
 * Input : ptm - Pointer to the matrix.
 */
void munit( matrix *ptm )
{
	register int i;

#if RUNCHK

	if (ptm->row != ptm->col)
		merror("matrix not quadratic error in mdiag!");

#endif

	minit(ptm);
	for ( i=0; i < ptm->row; i++ ) ptm->mat[i][i] = 1.0;
}


/*
 * MSET :
 * ------
 * This function copies the content of a matrix to another matrix.
 * ( A := B ) == mset(a,b); Assignment of matrix.
 * Inputs : ptm1, ptm2 - Pointers to matrices.
 */
void mset( matrix *ptm1, matrix *ptm2 )
{
	register int bytes;

#if RUNCHK

	if ( !( (ptm1->row == ptm2->row) && (ptm1->col == ptm2->col) ) ) \
	merror("Dimension mismatch error in mset!");

#endif

	bytes = 8*(ptm1->row)*(ptm1->col);
	memcpy( ptm1->mat[0], ptm2->mat[0], bytes );
}


/*
 * MADD
 * ----
 *
 * Matrix addition
 *
 * Input:  *ptm1 - Pointer to result matrix
 *                 (can be equal to *ptm1 and/or *ptm2)
 *         *ptm2 - Pointer to first argument matrix
 *         *ptm3 - Pointer to second argument matrix
 */
void madd( matrix *ptm1, matrix *ptm2, matrix *ptm3 )
{
	register int i,j;

#if RUNCHK

	if (!((ptm1->row == ptm2->row) && (ptm2->row == ptm3->row))) \
	merror("Dimension mismatch error in madd!");
	if (!((ptm1->col == ptm2->col) && (ptm2->col == ptm3->col))) \
	merror("Dimension mismatch error in madd!");

#endif

	/* Add the two matrices element by element */
	for ( i=0; i < ptm2->row; i++ )
	{
		for ( j=0; j < ptm2->col; j++ )
		{
			ptm1->mat[i][j] = ptm2->mat[i][j] + ptm3->mat[i][j];
		}
	}
}


/*
 * MSUB
 * ----
 *
 * Matrix subtraction
 *
 * Input:  *ptm1 - Pointer to result matrix
 *                 (can be equal to *ptm1 and/or *ptm2)
 *         *ptm2 - Pointer to first argument matrix
 *         *ptm3 - Pointer to second argument matrix
 */
void msub( matrix *ptm1, matrix *ptm2, matrix *ptm3 )
{
	register int i,j;

#if RUNCHK

	if (!((ptm1->row == ptm2->row) && (ptm2->row == ptm3->row))) \
	merror("Dimension mismatch error in msub!");
	if (!((ptm1->col == ptm2->col) && (ptm2->col == ptm3->col))) \
	merror("Dimension mismatch error in msub!");

#endif

	/* Subtract the two matrices element by element */
	for ( i=0; i < ptm2->row; i++ )
	{
		for ( j=0; j < ptm2->col; j++ )
		{
			ptm1->mat[i][j] = ptm2->mat[i][j] - ptm3->mat[i][j];
		}
	}
}


/*
 * MMUL
 * ----
 *
 * Matrix multiplication
 *
 * Input:  *ptm1 - Pointer to result matrix (Not equal to *ptm1 or *ptm2) 
 *         *ptm2 - Pointer to first argument matrix
 *         *ptm3 - Pointer to second argument matrix
 *
 */
void mmul( matrix *ptm1, matrix *ptm2, matrix *ptm3 )
{
	register int i,j,k;

#if RUNCHK

	if ((ptm1==ptm2) || (ptm1==ptm3)) \
	merror("Memory conflict error in mmul!");

	if ( !( (ptm2->col == ptm3->row) && \
	(ptm2->row == ptm1->row) && (ptm3->col == ptm1->col) ) ) \
	merror("Dimension mismatch error in mmul!");

#endif

	for ( i=0; i < ptm2->row; i++ )
	{
		for ( j=0; j < ptm3->col; j++ )
		{
			ptm1->mat[i][j] = 0.0;
			for ( k=0; k < ptm2->col; k++ )
			{
				ptm1->mat[i][j] += ptm2->mat[i][k] * ptm3->mat[k][j];
			}
		}
	}
}


/*
 * SMUL
 * ----
 *
 * Multiply matrix with scalar
 *
 * Input:  *ptm1  - Pointer to result matrix
 *         *ptm2  - Pointer to argument matrix
 *         factor - Scalar factor to be multiplied with argument matrix  
 */
void smul( matrix *ptm1, matrix *ptm2, double factor)
{
	register int i,j;

#if RUNCHK

	if (!((ptm1->row==ptm2->row) && (ptm1->col==ptm2->col))) \
	merror("Dimension mismatch error in smul!");

#endif

	for( i=0; i < ptm2->row; i++ )
	{
		for( j=0; j < ptm2->col; j++ )
		{
			ptm1->mat[i][j] = factor*ptm2->mat[i][j];
		}
	}
}


/*
 * TRACE :
 * -------
 * Calculates the trace (the sum of the main diagonal elements) of a matrix.
 * Input  : ptm   - Pointer to the matrix.
 * Output : trace - Trace of the matrix.
 */
double trace( matrix* ptm )
{
	register double trace = 0.0;
	register int i;

#if RUNCHK

	/* Check whether matrix is square */
	if (ptm->row != ptm->col) \
	merror("Matrix not square error in trace!");

#endif

	/* Calculate sum of diagonal elements */
	for ( i=0; i < ptm->row; i++)
	{
		trace += ptm->mat[i][i];
	}

	/* Return with trace */
	return trace;
}


/* SPROD :
 * -------
 * This function calculates the scalar-product of two vectors of equal
 * length. The vectors may be either row- or column- vectors or any 
 * combination of those. (The usual mmul routine can be used to calculate
 * scalar-products, but it requires a row- and a column-vector.)
 * Inputs : ptv1 - Pointer to first factor vector.
 *          ptv2 - Pointer to second factor vector.
 * Output : prod - Scalar product of the two vectors.
 */
double sprod( matrix* ptv1, matrix* ptv2 )
{
	register double prod = 0.0;
	register int i, elements;

#if RUNCHK

	/* Check whether the arguments are vectors.   */
	if (!(((ptv1->row==1)||(ptv1->col==1))&&((ptv2->row==1)||(ptv2->col==1)))) \
	merror("One of the inputs is not a vector error in sprod!");

	/* Check whether vectors has the same length. */
	if ((ptv1->row + ptv1->col) != (ptv2->row + ptv2->col)) \
	merror("Dimension mismatch error in sprod!");

#endif
	/* Calculate scalar product of the vectors.   */

	elements = ptv1->row + ptv1->col - 1;

	for ( i=0; i<elements; i++ ) {
		prod += (vget(ptv1,i))*(vget(ptv2,i));
	}

	return prod;
}


/* SPROD2 :
 * --------
 * This function calculates the scalar-product of two vectors of equal
 * length. The vectors may be either row- or column- vectors or any 
 * combination of those. (The usual mmul routine can be used to calculate
 * scalar-products, but it requires a row- and a column-vector.) In this
 * function, a starting and ending index can be specified for each vector.
 * In which case, only the specified part of the vectors will be used for
 * calculating the scalar-product.
 *
 * Inputs : ptv1 - Pointer to first factor vector.
 *          ptv2 - Pointer to second factor vector.
 *          begX - Beginning index of vectors 1 and 2.
 *          endX - Ending index of vectors 1 and 2.
 * Output : prod - Scalar product of the two vectors.
 */
double sprod2( matrix* ptv1, matrix* ptv2, int beg1, int end1, int beg2, int end2 )
{
	register double prod = 0.0;
	register int i, elements;

#if RUNCHK

	/* Check whether the arguments are vectors.   */
	if (!(((ptv1->row==1)||(ptv1->col==1))&&((ptv2->row==1)||(ptv2->col==1)))) \
	merror("One of the inputs is not a vector error in sprod!");

	/* Check whether vectors has the same length. */
	if ((end1 - beg1) != (end2 - beg2)) \
	merror("Dimension mismatch error in sprod!");

#else
	end2 = end2;  /* Avoid a warning. */
	
#endif
	/* Calculate scalar product of the vectors.   */

	elements = end1 - beg1 + 1;

	for ( i=0; i<elements; i++ ) {
		prod += (vget(ptv1,i+beg1))*(vget(ptv2,i+beg2));
	}

	return prod;
}

	
/*
 * MPUT :
 * ------
 * This function assigns a specified value to a specified element in
 * a matrix.
 * Inputs : ptm     - Pointer to the matrix.
 *          value   - Value to assign in matrix.
 *          row_pos - The row in which the element is to be assigned.
 *          col_pos - The column in which the element is to be assigned.
 */
void mput( matrix *ptm, int row_pos, int col_pos, double value )
{

#if RUNCHK

	/* Check if indices are inside matrix bounds. */
	if ( !((row_pos >= 0) && (row_pos < ptm->row)) ) \
	merror("Index out of range error in mput!");
	if ( !((col_pos >= 0) && (col_pos < ptm->col)) ) \
	merror("Index out of range error in mput!");

#endif

	/* Assign the value to the element */
	ptm->mat[row_pos][col_pos] = value;
}

	
/*
 * VPUT :
 * ------
 * This function assigns a specified value to a specified element in
 * a vector.
 * Inputs : ptv   - Pointer to the vector.
 *          value - Value to assign in matrix.
 *          pos   - The position in which the element is to be assigned.
 */
void vput( matrix *ptv, int pos, double value )
{

#if RUNCHK

	/* Check if index is inside vector bounds. */
	if ( !((pos>=0) && (pos <= ptv->row + ptv->col - 2)) ) \
	merror("Index out of range error in vput!");

#endif

	/* Assign the value to the element */
	if (ptv->row == 1)
	{
		ptv->mat[0][pos] = value; /* Row vector */
	}
	else if (ptv->col == 1)
	{
		ptv->mat[pos][0] = value; /* Column vector */
	}
	else merror("Input is not a vector error in vput!");

}



/*
 * MGET :
 * ------
 * This function returns a pointer to a specified element in a matrix.
 * Inputs : ptm     - Pointer to the matrix.
 *          row_pos - row of element.
 *          col_pos - column of element.
 */
double mget( matrix *ptm, int row_pos, int col_pos )
{

#if RUNCHK

	/* Check whether indices is inside matrix bounds. */
	if ( !((row_pos >= 0) && (row_pos < ptm->row)) ) \
	merror("Index out of range error in mget!");
	if ( !((col_pos >= 0) && (col_pos < ptm->col)) ) \
	merror("Index out of range error in mget!");

#endif

	/* Get address of specified element */
	return (ptm->mat[row_pos][col_pos]);
}


/*
 * VGET :
 * ------
 * Gets an element of either a column- or row-vector.
 * Inputs : ptv - Pointer to the vector.
 *          pos - Position of element.
 * Output : pte - Pointer to the specified element.
 */
double vget( matrix* ptv, int pos )
{
	register double *pte;

#if RUNCHK

	/* Check if index is inside matrix bounds. */
	if ( pos > ptv->row + ptv->col - 2 ) \
	merror("Index out of range error in vget!");

#endif

	if ( ptv->row == 1 )
	{
		pte = &(ptv->mat[0][pos]);
	}
	else if ( ptv->col == 1 )
	{
		pte = &(ptv->mat[pos][0]);
	}
	else merror("Input is not a vector in vget!");

	return *pte;
}


/*
 * ADDCOLS :
 * ---------
 * Create a new matrix, by putting two matrices next to one another
 *
 *                            [    :    ]
 *                       M1 = [ M2 : M3 ]
 *                            [    :    ]
 *
 * OBS: The number of rows in M1, M2 and M3 must be the same, and the number 
 *      of columns in M1 must be equal to the sum of rows in M2 and M3.
 *
 * Inputs:   *ptm1       : Pointer to result matrix (can be equal to *ptm2)
 *           *ptm2       : Pointer to the matrix to be placed left
 *           *ptm3       : Pointer to the matrix to be placed right
 */
void addcols(matrix *ptm1, matrix *ptm2, matrix *ptm3)
{
	register int i, j;

#if RUNCHK

	/* Check whether M1, M2 and M3 have the same number of rows        */
	if ( !( (ptm2->row == ptm3->row) && (ptm2->row == ptm1->row) ) ) \
	merror("Dimension mismatch in addcols!");
	
	/* Check that there aren't too many columns in M1                  */
	if ( ptm1->col != (ptm2->col + ptm3->col) ) \
	merror("Number of columns out of range in addcols!");

#endif

	/* Copy M2 to the left side of M1 */
	for( j = 0; j < ptm2->col; j++ )
	{
		for( i = 0; i < ptm1->row; i++ ) 
		{
			ptm1->mat[i][j] = ptm2->mat[i][j];
		}
	}

	/* Copy M3 to the right side of M1 */
	for( j = 0; j < ptm3->col; j++)
	{
		for( i = 0; i < ptm1->row; i++ )
		{
			ptm1->mat[i][(j+ptm2->col)] = ptm3->mat[i][j];
		}
	}

}

/*
 * ADDROWS :
 * ---------
 * Create a new matrix, by putting two matrices on top of one another
 *
 *                            [ M2 ]
 *                       M1 = [----]
 *                            [ M3 ]
 *
 * OBS: The number of columns in M2 and M3 must be the same, and the size of 
 *      M1 must fit M2 and M3.
 *
 * Inputs:   *ptm1       : Pointer to result matrix 
 *           *ptm2       : Pointer to matrix to be placed in the top
 *           *ptm3       : Pointer to matrix to be placed in the bottom
 */
void addrows( matrix *ptm1, matrix *ptm2, matrix *ptm3 )
{
	register int i, j;

#if RUNCHK
       
	/* Check whether M1, M2 and M3 have the same number of columns     */
	if (!((ptm2->col == ptm3->col) && (ptm1->col == ptm2->col))) \
	merror("Dimension mismatch in addrows!");

	/* Check whether M1 has the right dimensions to fit M2 & M3        */
	if (ptm1->row != (ptm2->row + ptm3->row)) \
	merror("Dimension mismatch in addrows!");

#endif

	/* Copy M2 to the top of M1 */
	for ( i = 0; i < ptm2->row; i++ )
	{
		for ( j = 0; j < ptm1->col; j++ ) 
		{
			ptm1->mat[i][j] = ptm2->mat[i][j];
		}
	}

	/* Copy M3 to the bottom of M1 */
	for ( i = 0; i < ptm3->row; i++ )
	{
		for ( j = 0; j < ptm1->col; j++ )
		{
			ptm1->mat[i+(ptm2->row)][j] = ptm3->mat[i][j];
		}
	}
}


/*
 * CONCAT :
 * --------
 * Concatenates parts of two vectors (either rows or columns). The 
 * part of each vector is specified by the beginning and ending index
 * of the part, which is to concatenated with the other.
 *
 * Inputs : ptv0 - Pointer to resulting vector.
 *          ptv1 - Pointer to first argument vector.
 *          ptv2 - Pointer to second argument vector.
 *          beg1 - Beginning index of first vector.
 *          end1 - Ending index of first vector.
 *          beg2 - Beginning index of second vector.
 *          end2 - Ending index of second vector.
 */
void concat(matrix* ptv0,matrix* ptv1,matrix* ptv2,int beg1,int end1,int beg2,int end2)
{
	register int i, n1, n2;

	n1 = end1 - beg1 + 1;   /* Number of elements in first vector.  */
	n2 = end2 - beg2 + 1;   /* Number of elements in second vector. */

#if RUNCHK

	/* Check that all arguments are vectors. */
	if (!(((ptv0->row==1)||(ptv0->col==1))&&((ptv1->row==1)||(ptv1->col==1))&&((ptv2->row==1)||(ptv2->col==1)))) \
	merror("Input is not a vector error in concat!");
	
	/* Check whether indices are inside bounds. */
	if ((end1 > (ptv1->row + ptv1->col - 2))||(end2 > (ptv2->row + ptv2->col - 2))) \
	merror("Index is outside vector bounds error in concat!");

	/* Check the dimension matching. */
	if ((n1 + n2) != (ptv0->row + ptv0->col - 1)) \
	merror("Dimension mismatch error in concat!");
#endif

	/* Copy part of first vector to beginning of result vector. */
	for ( i = 0; i < n1; i++ ) {
		vput( ptv0, i, vget( ptv1, i+beg1 ) );
	}

	/* Copy part of second vector to end of result vector. */
	for ( i = 0; i < n2; i++ ) {
		vput( ptv0, i+n1, vget( ptv2, i+beg2 ) );
	}
}


/*
 * MAT2MAT
 * _______
 *
 * Inserts a matrix (Matrix 2) into another matrix (Matrix 1).
 *
 *                 [      ]
 *            M1 = [  [M2]]
 *                 [      ]
 *
 * Inputs:   *ptm1      - Pointer Matrix 1
 *           row,col    - Coordinates to the element where the
 *                        upper left corner of matrix 2 is placed
 *           *ptm2      - Pointer to Matrix 2
 *
 */

void mat2mat( matrix *ptm1, int row, int col, matrix *ptm2 )
{
	register int i2,j2;
	int i1, j1, rows2, cols2;

	rows2 = ptm2->row;     /* Number of rows in Matrix 2    */
	cols2 = ptm2->col;     /* Number of columns in Matrix 2 */

#if RUNCHK
	/* Check whether the insertion is possible. */
	if( row+rows2 > ptm1->row || col+cols2 > ptm1->col )
		merror("Dimension mismatch in function MAT2MAT");
#endif

	/* Insert Matrix 2 into Matrix 1 */
	
	for( i1=row,i2=0; i2<rows2; i1++,i2++ )
	{
		for( j1=col,j2=0; j2<cols2; j1++,j2++  )
		{
			ptm1->mat[i1][j1] = ptm2->mat[i2][j2];
		}
	}
}



/*
 * SHIFT :
 * -------
 * Shifts all elements of a vector and inserts a new value 
 * in the first element. The last element is lost. The
 * function works on both column vectors and row vectors.
 *
 * Inputs:   *ptm        : Pointer to the vector to be shifted
 *           new_element : New element to be inserted
 */
void shift( matrix *ptm, double new_element)
{
	register int i,j;

	if (ptm->col == 1)                                         /* Column vector        */
	{
		for(i=ptm->row-2,j=i+1; j>0; i--,j--)              /* Shift elements down  */
			ptm->mat[j][0] = ptm->mat[i][0];
		ptm->mat[0][0] = new_element;                      /* Insert new element   */
	}
	else if (ptm->row == 1)                                    /* Row vector           */
	{
		for(i=ptm->col-2,j=i+1; j>0; i--,j--)              /* Shift elements right */
		{
			ptm->mat[0][j] = ptm->mat[0][i];
		}
		ptm->mat[0][0] = new_element;                      /* Insert new element   */
	}
	else merror("Dimension mismatch error in funcion shift!"); /* Not a vector         */      
}


/*
 * SUBMAT
 * ------
 * This function picks a submatrix from a larger matrix.
 *
 * Inputs:        *ptm1   - Pointer to the result matrix 
 *                *ptm2   - Pointer to input matrix
 *                rowbeg  - Number of the first row
 *                rowend  - Number of the last row
 *                colbeg  - Number of the first column
 *                colend  - Number of the last column
 */
void submat( matrix *ptm1, matrix *ptm2, int rowbeg, int rowend, int colbeg, int colend )
{
	register int i1,i2,j1,j2;

#if RUNCHK

	if ( rowend > ptm2->row-1 || colend > ptm2->col-1 ) \
	merror("Argument out of range in function submat!");
	
	if (((rowend-rowbeg)!=(ptm1->row-1))||((colend-colbeg)!=(ptm1->col-1)))\
	merror("Dimension mismatch error in function submat!");

#endif

	for ( i1=0,i2=rowbeg; i2<=rowend; i1++,i2++ )
	{
		for ( j1=0,j2=colbeg; j2<=colend; j1++,j2++ ) \
		ptm1->mat[i1][j1] = ptm2->mat[i2][j2];
	}
}


/*
 * SUBVEC
 * ------
 *
 * Inputs:        *ptm1   - Pointer to the result vector
 *                *ptm2   - Pointer to input vector
 *                elem1   - Number of the first element
 *                elem2   - Number of the last element
 */
void subvec( matrix *ptm1, matrix *ptm2, int elem1, int elem2 )
{
	register int i,j;

#if RUNCHK

	/* Check whether elem2 is inside bound */
	if ( elem2 > ptm2->row + ptm2->col - 2 ) \
	merror("Argument out of range in function subvec!");

	/* Check whether sub-vector fits output-vector. */
	if ((ptm1->row + ptm1->col) != ( elem2 - elem1 + 2 )) \
	merror("Wrong number of elements error in subvec!");


#endif

	if ( ptm2->col == 1 )                                 /* Column vector */
	{
		for ( i=0,j=elem1; j<=elem2; i++,j++ ) \
		ptm1->mat[i][0] = ptm2->mat[j][0];
	}

	else if ( ptm2->row == 1 )                            /* Row vector    */
	{
		for ( i=0,j=elem1; j<=elem2; i++,j++ ) \
		ptm1->mat[0][i] = ptm2->mat[0][j];
	}
	else merror("Dimension mismatch in function subvec"); /* Not a vector  */
}


/*
 * GETROWS
 * -------
 *
 * This function gets the number of rows in a matrix
 *
 * Input:   *ptm - Pointer to matrix
 *
 * Output:  rows - Number of rows in matrix
 *
 */
int getrows(matrix *ptm)
{
	return ptm->row;
}



/*
 * GETCOLS
 * -------
 *
 * This function gets the number of columns in a matrix
 *
 * Input:   *ptm - Pointer to matrix
 *
 * Output:  columns - Number of columns in matrix
 *
 */
int getcols(matrix *ptm)
{
	return ptm->col;
}


/*
 * LENGTH :
 * --------
 * Determines the length of a row- or column- vector.
 * Input  : ptv - Pointer to the vector.
 * Output : length - Length of the vector. (The number of elements.)
 */
int length( matrix* ptv )
{
	register int length;

#if RUNCHK

	if ( !( (ptv->row==1) || (ptv->col==1) ) ) \
	merror("Input argument is not a vector error in length!");

#endif
	length = ptv->row + ptv->col - 1;

	return length;
}


/*
 * MSCMP
 * -----
 *
 * This function compares a string and a row vector containing a 
 * MATLAB string loaded by mload.
 * If the to strings are identical 1 is returned. Otherwise the 
 * function returns 0.
 *
 * Input:  mat    -  Pointer to "string vector"
 *         string -  String expression
 *
 */

int mscmp( matrix *mat, char *string )
{
	int i, matlength;

	matlength=length(mat);
	for( i=0 ; (char)get_val(mat,0,i) == *(string+i) && i<matlength ; i++ );
	if( (i==matlength) && *(string+i)=='\0') return 1;
	else return 0;
}



#ifndef NO_IO  /* For compilation by the DSP ANSI-C compiler. */

/*
 * loadmat - C language routine to load a matrix from a MAT-file.
 *
 * Here is an example that uses 'loadmat' to load a matrix from a MAT-file:
 *
 *      FILE *fp;
 *      char name[20];
 *      int type, mrows, ncols, imagf;
 *      double *xr, *xi;
 *      fp = fopen("foo.mat","rb");
 *      loadmat(fp, &type, name, &mrows, &ncols, &imagf, &xr, &xi);
 *      fclose(fp);
 *      free(xr);
 *      if (imagf) free(xi);
 *
 * The 'loadmat' routine returns 0 if the read is successful and 1 if
 * and end-of-file or read error is encountered.  'loadmat' can be called
 * repeatedly until the file is exhausted and an EOF is encountered.
 *
 * Author J.N. Little 11-3-86
 *
 * Modified by Steffen Torp, Servolaboratoriet 14-10-92.
 */

int loadmat(FILE *fp,int *type,char *pname,int *mrows,int *ncols,int *imagf,double **preal,double **pimag)
/* FILE *fp;        File pointer */
/* int *type;       Type flag: see reference section of guide */
/* int *mrows;      row dimension */
/* int *ncols;      column dimension */
/* int *imagf;      imaginary flag */
/* char *pname;     pointer to matrix name */
/* double **preal;  pointer to real data */
/* double **pimag;  pointer to imag data */
{
	Fmatrix x;
	int mn, namlen, conv;
	

	/*
	 * Get Fmatrix structure from file
	 */
	if (fread((char *)&x, sizeof(Fmatrix), 1, fp) != 1) {
		return(1);
	}


	/* Check if the mat-file is created on a platform similar to */
	/* the actual platform. If yes, the flag 'conv' will be FALSE*/
	/* Otherwise it will be TRUE                                 */
	
	if (SYSTEM == 0) {
		switch (x.type) {
			case 0L:          conv = FALSE; break;
			case 1L:          conv = FALSE; break;
			case 0xE8030000L: conv = TRUE;  break;
			case 0xE9030000L: conv = TRUE;  break;
			default: {
				printf("\nUnknown file type in loadmat!\n");
				printf("The type is: %lx \n",x.type);
				exit(1);
			}
		}
	}
	else if (SYSTEM == 1000) {
		switch (x.type) {
			case 0L:         conv = TRUE;  break;
			case 0x1000000L: conv = TRUE;  break;
			case 1000L:      conv = FALSE; break;
			case 1001L:      conv = FALSE; break;
			default: {
				printf("\nUnknown file type in loadmat!\n");
				printf("The type is: %lx \n",x.type);
				exit(1);
			}
		}
	}

	if(conv==TRUE)
	{
		*type  = (int)longconv(x.type);
		*mrows = (int)longconv(x.mrows);
		*ncols = (int)longconv(x.ncols);
		*imagf = (int)longconv(x.imagf);
		namlen = (int)longconv(x.namlen);
		mn     = (*mrows)*(*ncols);
	}
	else
	{
		*type  = (int)(x.type);
		*mrows = (int)(x.mrows);
		*ncols = (int)(x.ncols);
		*imagf = (int)(x.imagf);
		namlen = (int)(x.namlen);
		mn     = (*mrows)*(*ncols);
	}
       
	/*
	 * Get matrix name from file
	 */
	if (fread(pname, sizeof(char), namlen, fp) != namlen) {
		return(1);
	}
	
	
	/*
	 * Get Real part of matrix from file
	 */
	if (mn) {
		*preal = (double *)malloc(mn*sizeof(double));
		if (!(*preal)) {
			printf("\nError: Variable too big to load\n");
			return(1);
		}
		if (fread(*preal, sizeof(double), mn, fp) != mn) {
			free(*preal);
			return(1);
		}
		if(conv==TRUE) matconv(mn,*preal);

	}

	/*
	 * Get Imag part of matrix from file, if it exists
	 */
	if (x.imagf) {
		*pimag = (double *)malloc(mn*sizeof(double));
		if (!(*pimag)) {
			printf("\nError: Variable too big to load\n");
			free(*preal);
			return(1);
		}
		if (fread(*pimag, sizeof(double), mn, fp) != mn) {
			free(*pimag);
			free(*preal);
			return(1);
		}
		if(conv==TRUE) matconv(mn,*pimag);
	}
	return(0);
}

/*
 * savemat - C language routine to save a matrix in a MAT-file.
 *
 * Here is an example that uses 'savemat' to save two matrices to disk,
 * the second of which is complex:
 *
 *      FILE *fp;
 *      double xyz[1000], ar[1000], ai[1000];
 *      fp = fopen("foo.mat","wb");
 *      savemat(fp, 2000, "xyz", 2, 3, 0, xyz, (double *)0);
 *      savemat(fp, 2000, "a", 5, 5, 1, ar, ai);
 *      fclose(fp);
 *
 * Author J.N. Little 11-3-86
 */

void savemat(FILE *fp,int type,char *pname,int mrows,int ncols,int imagf,double *preal,double *pimag)
/* FILE *fp;      File pointer */
/* int type;      Type flag: Normally 0 for PC, 1000 for Sun, Mac, and  */
/*                Apollo, 2000 for VAX D-float, 3000 for VAX G-float    */
/*                Add 1 for text variables.      */
/*                See LOAD in reference section of guide for more info. */
/* int mrows;     row dimension */
/* int ncols;     column dimension */
/* int imagf;     imaginary flag */
/* char *pname;   pointer to matrix name */
/* double *preal; pointer to real data */
/* double *pimag; pointer to imag data */
{
	Fmatrix x;
	int mn;
	
	x.type   = (long)type;
	x.mrows  = (long)mrows;
	x.ncols  = (long)ncols;
	x.imagf  = (long)imagf;
	x.namlen = (long)strlen(pname) + 1L;
	mn       = (int)(x.mrows * x.ncols);

	fwrite(&x, sizeof(Fmatrix), 1, fp);
	fwrite(pname, sizeof(char), (int)x.namlen, fp);
	fwrite(preal, sizeof(double), mn, fp);
	if (imagf) {
	     fwrite(pimag, sizeof(double), mn, fp);
	}
}


/* 
 * LONGCONV
 * --------
 *
 * This function flips long integers, in order to convert longs
 * from INTEL to MOTOROLA format or vice versa.
 *
 * INPUT  :   inval  - Long integer to be converted
 *
 * OUTPUT :   outval - The converted long.
 *
 */

long longconv(long inval)
{
	char *ptc1, *ptc2;
	int i, j;
	long outval;
	ptc1=(char*)(&inval);
	ptc2=(char*)(&outval);
	for(i=0,j=3;i<4;i++,j--)
	{
		*(ptc2+i)=*(ptc1+j);
	}
	return(outval);
}


/* 
 * MATCONV
 * -------
 *
 * This function flips a number of doubles, in a consecutive part 
 * of the memory, in order to convert matrix data from INTEL to  
 * MOTOROLA format or vice versa.
 *
 * INPUT  :   mn     - Number of elements in the matrix
 *            ptd    - Pointer to the first element
 * 
 * 
 */

void matconv(int mn, double *ptd)
{
	char *ptc;
	char tmp;
	int i, j, k;
	for(k=0;k<mn;k++)
	{
		/* Flip the k'th double */
		ptc=(char*)(ptd+k);
		for(i=0,j=7;i<4;i++,j--)
		{
			tmp=*(ptc+i);
			*(ptc+i)=*(ptc+j);
			*(ptc+j)=tmp;
		}
	}
}


#endif /* I/O functions are not available in the DSP compiler. */

